//	CDialogNewDisk.c

#include "CDisk.h"
#include "ADFS_Menus.h"
#include "ADFS_Prefs.h"
#include "ProStructs.h"
#include "Utils.h"
#include "CDialogNewDisk.h"

CDialogNewDisk		*gNewDiskP = NULL;

CDialogNewDisk		*ShowNewDisk(ADFS_NewDiskRec *newDiskP)
{
	ASSERT(!gNewDiskP);
	
	if (!gNewDiskP) {
		gNewDiskP = new CDialogNewDisk;
		
		if (gNewDiskP) {
			if (!gNewDiskP->IDialogNewDisk(newDiskP)) {
				gNewDiskP->Dispose();
				gNewDiskP = NULL;
			}
		}
	}

	return gNewDiskP;
}

CDialogNewDisk		*GetNewDisk(void)
{
	return gNewDiskP;
}

/*********************************************/
#define		kDiskDlogWidth			247
#define		kDiskDlogHeight			232

#define		kSectorDlogPopOffset		30
#define		kSectorDlogButtonsOffset	90
#define		kSectorDlogHeight			(kDiskDlogHeight - kSectorDlogButtonsOffset)

Boolean		CDialogNewDisk::IDialogNewDisk(ADFS_NewDiskRec *newDiskP)
{
	OSErr				err = noErr;
	
	//	get it hidden, then show
	if (!_inherited::IDialog(ADFS_Window_NEW_DISK, 300)) {
		err = IC_Err_COULDNT_LOAD_DLG;
	}

	if (!err) {
		WindowRef	windowRef = GetWindowRef();
		Rect		windR = (**gPrefsH).windA[PREF_Wind_NEW_DISK];

		i_newDiskRec = *newDiskP;
		
		if (!EmptyRect(&windR)) {
			MoveWindow(
				windowRef, 
				windR.left, 
				windR.top, 
				FALSE);
		}
		
		i_origDiskFileType	= i_newDiskRec.format.diskFileType;
		
		i_showSizePopB		= TRUE;
		i_showSizeUnitsB	= TRUE;
		i_showNibblizeB		= TRUE;
		i_showSectorB		= TRUE;
		
		SetDialogDefaultItem(GetDialogRef(), ADFS_NewDisk_OKAY);

		ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_CHECK_NIBBLIZED);
		ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_SECTOR_ORDER);
		ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_FORMAT);
		
		if (i_newDiskRec.diskP) {
			i_showSizePopB		= FALSE;
			i_showSizeUnitsB	= FALSE;

			HideDialogItemGroup(
				ADFS_NewDisk_DISK_Gr_FIRST, 
				ADFS_NewDisk_DISK_Gr_LAST);

			OffsetDlogItem(
				ADFS_NewDisk_POP_SECTOR_ORDER, 0, -kSectorDlogPopOffset);

			OffsetDlogItem(
				ADFS_NewDisk_CHECK_NIBBLIZED, 0, -kSectorDlogPopOffset);
			
			//	just get it out of the way
			OffsetDlogItem(
				ADFS_NewDisk_POP_VOL_SIZE, 0, kSectorDlogPopOffset);

			OffsetDlogItem(ADFS_NewDisk_OKAY,	0, -kSectorDlogButtonsOffset);
			OffsetDlogItem(ADFS_NewDisk_CANCEL,	0, -kSectorDlogButtonsOffset);
			
			SizeWindow(
				windowRef, kDiskDlogWidth, kSectorDlogHeight, FALSE);
			
			if (i_newDiskRec.disk2imgB) {
				SetTitle("Disk to Image");
			} else {
				SetTitle("Disk Format");
			}
		} else {
			ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_OS);
			ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_CHECK_BOOTABLE);
			ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_VOL_SIZE);
			ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_SIZE_TE);
			ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_SIZE_UNITS);
		}
				
		ShowWindow(windowRef);
	}

	if (!err) {
		SetModalUp(TRUE);
	}

	return err == noErr;
}

void		CDialogNewDisk::Dispose(void)
{
	_inherited::Dispose();
	gNewDiskP = NULL;
	SetModalUp(FALSE);
}

void		CDialogNewDisk::Finish(void)
{
	if (i_newDiskRec.format.nibblizedB) {
		i_newDiskRec.format.sectorOrder = FSType_C2P;
	}
	
	if (i_newDiskRec.diskP && !i_newDiskRec.disk2imgB) {	
		if (i_newDiskRec.diskP->SetDiskFormat(&i_newDiskRec.format)) {
			Dispose();
		}
	} else {
		long		oldBytesOnDiskL = i_newDiskRec.bytesOnDiskL;
		
		ASSERT(i_newDiskRec.NewDiskDoneCB);
		HideWindow(GetWindowRef());
		
		if (oldBytesOnDiskL > Pro_kBlocksPerDisk32M * Pro_kBytesPerBlock) {
			i_newDiskRec.bytesOnDiskL = Pro_kBlocksPerDisk32M * Pro_kBytesPerBlock;
		}
		
		if ((*i_newDiskRec.NewDiskDoneCB)(&i_newDiskRec)) {
			Dispose();
		} else {
			i_newDiskRec.bytesOnDiskL = oldBytesOnDiskL;
			ShowWindow(GetWindowRef());
		}
	}
}

void		CDialogNewDisk::Move(EventRecord *event)
{
	_inherited::Move(event);
	(**gPrefsH).windA[PREF_Wind_NEW_DISK] = GetWindowRect(WindowRect_ALL_GLOBAL);
	SavePrefs();
}

Boolean		CDialogNewDisk::DoCommand(long command)
{
	Boolean			handled = FALSE;
	
	switch (command) {

		case cmdCut: {
			DialogCut(GetDialogRef());
			break;
		}
		
		case cmdCopy: {
			DialogCopy(GetDialogRef());
			break;
		}
		
		case cmdPaste: {
			TERestoreRec	restoreRec;
			short			itemHit = GetDlogTextRestore(&restoreRec);
			
			DialogPaste(GetDialogRef());
			if (VerifyText(itemHit, &restoreRec)) {
				ValueConvert(DGI_ValueConvert_FROM, ADFS_NewDisk_SIZE_TE);
			}
			break;
		}
		
		case cmdClear: {
			DialogDelete(GetDialogRef());
			break;
		}
	}
	
	return handled;
}

void		CDialogNewDisk::DoKeyDown(EventRecord *theEvent)
{
	short			keyResult = DGI_KeyDown_NONE;
	DialogPtr		theDialog;
	char			theKey	= VirtualASCII(theEvent);
	short			itemHit;
	TERestoreRec	restoreRec;

	if (theEvent->modifiers & cmdKey) {
		
		//	cut, copy, paste, clear all get handled at the top
			
		switch (theKey) {
			
			case '.': {
				keyResult = DGI_KeyDown_EXIT;
				break;
			}
				
			default: {
				_inherited::DoKeyDown(theEvent);
				break;
			}
		}
	} else {

		switch (theKey) {
			
			case RETURN_KEY:
			case ENTER_KEY: {
				itemHit = GetDialogKeyboardFocusItem(GetDialogRef());
				keyResult = DGI_KeyDown_DONE;
				break;
			}
			
			case ESC_CLEAR_KEY: {
				itemHit = GetDialogKeyboardFocusItem(GetDialogRef());
				keyResult = DGI_KeyDown_EXIT;
				break;
			}
			
			case DELETE_KEY:
			case TAB_KEY:
			case LEFT_ARROW_KEY:
			case RIGHT_ARROW_KEY: {
				short		ignore;

				itemHit = GetDlogTextRestore(&restoreRec);
				DialogSelect(theEvent, &theDialog, &ignore);				
				keyResult = DGI_KeyDown_GOOD;
				break;
			}
			
			default: {
				if (IS_DIGIT(theKey)) {
					short		ignore;

					itemHit = GetDlogTextRestore(&restoreRec);
					DialogSelect(theEvent, &theDialog, &ignore);
					keyResult = DGI_KeyDown_GOOD;
				} else {
					keyResult = DGI_KeyDown_BAD;
				}
				break;
			}
		}		
	}
	
	switch (keyResult) {
		
		case DGI_KeyDown_NONE: {
			//	yaaay!
			break;
		}

		case DGI_KeyDown_EXIT: {
			SimulateClick(ADFS_NewDisk_CANCEL);
			Dispose();
			break;
		}

		case DGI_KeyDown_BAD: {
			SysBeep(1);
			break;
		}

		case DGI_KeyDown_GOOD: {			
			if (VerifyText(itemHit, &restoreRec)) {
				ValueConvert(DGI_ValueConvert_FROM, ADFS_NewDisk_SIZE_TE);
			}
			break;
		}

		case DGI_KeyDown_DONE: {
			SimulateClick(ADFS_NewDisk_OKAY);
			Finish();
			break;
		}
	}
}

/*******************/
void		CDialogNewDisk::DoClick(EventRecord *theEvent)
{
	if (FrontWindow() != GetWindowRef()) {
		_inherited::DoClick(theEvent);
	} else {
		DialogRef		theDialog;
		short			itemHit;
		ControlRef		controlRef;
		
		//SetDialogFont();
		
		DialogSelect(theEvent, &theDialog, &itemHit);
		GlobalToLocal(&theEvent->where);

		controlRef = GetDlogItemControl(itemHit);

		switch (itemHit) {
		
			case ADFS_NewDisk_OKAY: {
				Finish();
				break;
			}
			
			case ADFS_NewDisk_CANCEL: {
				Dispose();
				break;
			}
				
			case ADFS_NewDisk_CHECK_NIBBLIZED: {
				i_newDiskRec.format.nibblizedB = !i_newDiskRec.format.nibblizedB;
				ValueConvert(DGI_ValueConvert_TO, itemHit);
				ValueConvert(DGI_ValueConvert_FROM, itemHit);
				break;
			}
			
			case ADFS_NewDisk_CHECK_BOOTABLE: {
				i_newDiskRec.bootableB = !i_newDiskRec.bootableB;
				ValueConvert(DGI_ValueConvert_TO, itemHit);
				ValueConvert(DGI_ValueConvert_FROM, itemHit);
				break;
			}

			case ADFS_NewDisk_SIZE_TE: {
				//	ValueConvert(DGI_ValueConvert_FROM, itemHit);
				//	done elsewhere
				break;
			}
				
			case ADFS_NewDisk_SIZE_UNITS: {
				ValueConvert(DGI_ValueConvert_FROM, itemHit);
				ValueConvert(DGI_ValueConvert_TO, itemHit);
				break;
			}

			default: {
				ValueConvert(DGI_ValueConvert_FROM, itemHit);
				break;
			}			
		}
	}
}

void		CDialogNewDisk::UpdateMenus(void)
{
	EnableCommand(cmdCut);
	EnableCommand(cmdCopy);
	EnableCommand(cmdPaste);
	EnableCommand(cmdClear);
}

Boolean		CDialogNewDisk::VerifyText(short itemHit, TERestoreRec *restoreRec0)
{
//	ushort		newValueS	= 0;
	Boolean		badB		= FALSE;
	char		itemTextA[256];
	short		strLenS;
	
	itemTextA[0] = 0;
	GetDlogItemText(itemHit, itemTextA);
	strLenS = strlen(itemTextA);
	
	//	empty is okay, means zero
	if (strLenS != 0) {
		short		loop;
		
		if (i_newDiskRec.isMB) {
			loop = 2;	//	2 digits allowed
		} else {
			loop = 5;	//	5 digits allowed
		}
		
		badB = strLenS > loop;
		
		if (!badB) {
		
			if (loop > strLenS) {
				loop = strLenS;
			}
			
			//	must be all digits
			for (; !badB && loop > 0; loop--) {
				badB = !IS_DIGIT(itemTextA[loop - 1]);
			}
		}
		
		if (!badB) {
			ushort		value;
			//	all digits so far
	
			sscanf(itemTextA, "%hu", &value);
			
			if (i_newDiskRec.isMB) {
				badB = value > 32;
			} else {
				badB = value > 32768;
			}
		}
		
		if (badB) {
			SysBeep(1);
			if (restoreRec0) {
				RestoreDlogText(restoreRec0);
			}
		}
	}
	
	return !badB;
}

/*******************/
ND_DiskFileType	ND_DiskTypeToFileType(DiskType diskType)
{
	ND_DiskFileType	diskFileType;
	
	if (
		diskType == DiskType_inMemory_Raw
		|| diskType == DiskType_inMemory_Nib
		|| diskType == DiskType_onDisk_Raw
	) {
		diskFileType = ND_Format_RAW;
	} else if (IS_DiskType_2img(diskType)) {
		diskFileType = ND_Format_2IMG;
	} else {
		ASSERT(IS_DiskType_DiskCopy(diskType));

		diskFileType = ND_Format_DISK_COPY;
	}
	
	return diskFileType;
}

void		ImageRecToDiskFormatRec(DiskImageRec *imageRecP, ND_DiscFormatRec *formatRecP)
{
	formatRecP->sectorOrder		= ImageRec_OrigOrder(imageRecP);
	formatRecP->nibblizedB		= IS_ImageRec_NIB(imageRecP);
	formatRecP->diskFileType	= ND_DiskTypeToFileType(ImageRec_DiskType(imageRecP));
}

FSType		ND_OSToSector(FSType osType)
{
	FSType		sectorType;
	
	switch (osType) {
		case FSType_HYB:
		case FSType_DOS:	sectorType = FSType_DOS;	break;
		case FSType_PRO:	sectorType = FSType_PRO;	break;
		case FSType_PAS:	sectorType = FSType_PRO;	break;
		case FSType_CPM:	sectorType = FSType_CPM;	break;
		default:			sectorType = FSType_PRO;	break;
	}
	
	return sectorType;
}

static		FSType		ND_SectorToOS(FSType sectorType)
{
	FSType		osType;
	
	switch (sectorType) {
		case FSType_DOS:	osType = FSType_DOS;	break;
		case FSType_PRO:	osType = FSType_PRO;	break;
		case FSType_PAS:	osType = FSType_PRO;	break;
		case FSType_CPM:	osType = FSType_CPM;	break;
		default:			osType = FSType_PRO;	break;
	}
	
	return osType;
}

/*******************/
static		short		ND_TypeToMenu_Sector(FSType osType)
{
	short		menuItem;
	
	switch (osType) {
		default:
		case FSType_DOS:	menuItem = 1;	break;
		case FSType_PRO:	menuItem = 2;	break;
		case FSType_CPM:	menuItem = 3;	break;
		case FSType_C2P:	menuItem = 4;	break;
		case FSType_HYB:	menuItem = 6;	break;
	}
	
	return menuItem;
}

static		FSType		ND_MenuToType_Sector(short menuItem)
{
	FSType		sectorType;
	
	switch (menuItem) {
		default:
		case 1:		sectorType = FSType_DOS;	break;
		case 2:		sectorType = FSType_PRO;	break;
		case 3:		sectorType = FSType_CPM;	break;
		case 4:		sectorType = FSType_C2P;	break;
		case 6:		sectorType = FSType_DOS;	break;
	}
	
	return sectorType;
}

/*******************/
static		short		ND_TypeToMenu_OS(FSType osType)
{
	short		menuItem;
	
	switch (osType) {
		default:
		case FSType_DOS:	menuItem = 1;	break;
		case FSType_PRO:	menuItem = 2;	break;
		case FSType_PAS:	menuItem = 3;	break;
		case FSType_CPM:	menuItem = 4;	break;
		case FSType_HYB:	menuItem = 6;	break;
	}
	
	return menuItem;
}

static		FSType		ND_MenuToType_OS(short menuItem)
{
	FSType		osType;
	
	switch (menuItem) {
		default:
		case 1:		osType = FSType_DOS;	break;
		case 2:		osType = FSType_PRO;	break;
		case 3:		osType = FSType_PAS;	break;
		case 4:		osType = FSType_CPM;	break;
		case 6:		osType = FSType_HYB;	break;
	}
	
	return osType;
}

/*******************/
static		short		ND_TypeToMenu_Units(Boolean isMB)
{
	return 1 + isMB;
}

static		Boolean		ND_MenuToType_Units(short menuItem)
{
	return menuItem == 2;
}

/*******************/
enum {
	ND_140k		= 1, 
	ND_800k,
	ND_1MB, 
		ND_sep1, 
	ND_5MB, 
	ND_10MB, 
	ND_15MB, 
	ND_20MB, 
	ND_32MB, 
		ND_sep2, 
	ND_Custom
};

#define		kVolSizeCustom		-1L

static		long		ND_MenuToType_VolSize(short menuItem)
{
	long		bytesOnDiskL = 0;
	
	switch (menuItem) {
		
		default: {
			ASSERT(0);
			break;
		}
	
		case ND_Custom: {
			bytesOnDiskL = kVolSizeCustom;
			break;
		}

		case ND_140k: {
			bytesOnDiskL = Pro_kBlocksPerDisk * Pro_kBytesPerBlock;
			break;
		}

		case ND_800k: {
			bytesOnDiskL = Pro_kBlocksPerDisk800 * Pro_kBytesPerBlock;
			break;
		}

		case ND_1MB: {
			bytesOnDiskL = Pro_kBlocksPerDisk1440 * Pro_kBytesPerBlock;
			break;
		}

		case ND_5MB: {
			bytesOnDiskL = Pro_kBlocksPerDisk5M * Pro_kBytesPerBlock;
			break;
		}

		case ND_10MB: {
			bytesOnDiskL = Pro_kBlocksPerDisk10M * Pro_kBytesPerBlock;
			break;
		}

		case ND_15MB: {
			bytesOnDiskL = Pro_kBlocksPerDisk15M * Pro_kBytesPerBlock;
			break;
		}

		case ND_20MB: {
			bytesOnDiskL = Pro_kBlocksPerDisk20M * Pro_kBytesPerBlock;
			break;
		}

		case ND_32MB: {
			bytesOnDiskL = Pro_kBlocksPerDisk32M * Pro_kBytesPerBlock;
			break;
		}
	}
	
	return bytesOnDiskL;
}

static		short		ND_TypeToMenu_VolSize(long bytesOnDiskL)
{
	short	menuItem;
	
	for (menuItem = ND_140k; menuItem <= ND_32MB; menuItem++) {
		if (
			menuItem != ND_sep1
			&& menuItem != ND_sep2
		) {
			if (ND_MenuToType_VolSize(menuItem) == bytesOnDiskL) {
				return menuItem;
			}
		}
	}
	
	return ND_Custom;
}

void		CDialogNewDisk::ValueConvert(
	DGI_ValueConvertType	convertType, 
	short					dlgItem)
{
	ControlRef		controlRef = GetDlogItemControl(dlgItem);
//	Boolean			convertFromB = FALSE;
	
	switch (dlgItem) {
		
		case ADFS_NewDisk_POP_FORMAT: {
			if (convertType == DGI_ValueConvert_TO) {
				SetControlValue(
					controlRef, i_newDiskRec.format.diskFileType);
			} else {
				i_newDiskRec.format.diskFileType = (ND_DiskFileType)
					GetControlValue(controlRef);
				
				/*
					sets ADFS_NewDisk_CHECK_NIBBLIZED
					sets ADFS_NewDisk_POP_OS
					sets ADFS_NewDisk_POP_SECTOR_ORDER

					affects ADFS_NewDisk_POP_VOL_SIZE
					affects ADFS_NewDisk_SIZE_TE
					affects ADFS_NewDisk_SIZE_UNITS
				*/

				if (i_newDiskRec.format.diskFileType == ND_Format_DISK_COPY) {

					if (i_newDiskRec.format.nibblizedB) {
						i_newDiskRec.format.nibblizedB = FALSE;
						ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_CHECK_NIBBLIZED);
					}

					if (i_newDiskRec.osType != FSType_PRO) {
						i_newDiskRec.osType = FSType_PRO;
						ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_OS);
					}

					if (i_newDiskRec.format.sectorOrder != FSType_PRO) {
						i_newDiskRec.format.sectorOrder = FSType_PRO;
					}
				}

				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_SECTOR_ORDER);
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_VOL_SIZE);
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_SIZE_TE);
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_SIZE_UNITS);
			}
			break;
		}
		
		case ADFS_NewDisk_CHECK_NIBBLIZED: {
						
			if (convertType == DGI_ValueConvert_TO) {
				Boolean		isHybridB	= i_newDiskRec.osType		== FSType_HYB;
				Boolean		is140kB		= i_newDiskRec.bytesOnDiskL == Gen_kBytesPerDisk;

				SetControlValue(
					controlRef, 
					i_newDiskRec.format.nibblizedB);

				/*
					depends ADFS_NewDisk_POP_OS
					depends bytesOnDiskL
					
					can only show if
						OSType != hybrid
						bytesOnDiskL == 140k
				*/

				if (
					i_showNibblizeB
					&& (isHybridB || !is140kB)
				) {
					i_showNibblizeB = FALSE;
					HideDlogItem(ADFS_NewDisk_CHECK_NIBBLIZED);
					i_newDiskRec.format.nibblizedB = FALSE;
				} else if (
					!i_showNibblizeB
					&& !isHybridB 
					&& is140kB
				) {
					i_showNibblizeB = TRUE;
					ShowDlogItem(ADFS_NewDisk_CHECK_NIBBLIZED);
				}
			} else {
				/*
					sets	ADFS_NewDisk_POP_FORMAT
					
					affects ADFS_NewDisk_POP_VOL_SIZE
					affects ADFS_NewDisk_SIZE_TE
					affects ADFS_NewDisk_SIZE_UNITS
				*/
				//	kept up to date
				// i_newDiskRec.format.nibblizedB = GetControlValue(controlRef);
				
				if (i_newDiskRec.format.nibblizedB) {
					
					if (i_newDiskRec.format.diskFileType == ND_Format_DISK_COPY) {
						if (i_origDiskFileType == ND_Format_DISK_COPY) {
							i_newDiskRec.format.diskFileType = ND_Format_RAW;
						} else {
							i_newDiskRec.format.diskFileType = i_origDiskFileType;
						}
						
						ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_FORMAT);
					}
				}
				
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_VOL_SIZE);
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_SECTOR_ORDER);
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_SIZE_TE);
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_SIZE_UNITS);
			}
			break;
		}

		case ADFS_NewDisk_CHECK_BOOTABLE: {
			if (convertType == DGI_ValueConvert_TO) {
				SetControlValue(
					controlRef, 
					i_newDiskRec.bootableB);
				
				//	doesn't depend
			} else {
				//	kept up to date
				// i_newDiskRec.bootableB = GetControlValue(controlRef);

				//	doesn't affect
			}
			break;
		}

		case ADFS_NewDisk_POP_SECTOR_ORDER: {
			if (convertType == DGI_ValueConvert_TO) {

				SetControlValue(
					controlRef, 
					ND_TypeToMenu_Sector((FSType)i_newDiskRec.format.sectorOrder));

				/*
					depends ADFS_NewDisk_CHECK_NIBBLIZED
					depends bytesOnDiskL

					visible if
						not ADFS_NewDisk_CHECK_NIBBLIZED
						&& bytesOnDiskL == 140k
				*/

				if (
					!i_showSectorB
					&& i_newDiskRec.bytesOnDiskL == Gen_kBytesPerDisk
					&& !i_newDiskRec.format.nibblizedB
				) {
					i_showSectorB = TRUE;
					ShowDlogItem(ADFS_NewDisk_POP_SECTOR_ORDER);
				} else if (
					i_showSectorB
					&& i_newDiskRec.bytesOnDiskL != Gen_kBytesPerDisk
					|| i_newDiskRec.format.nibblizedB
				) {
					i_showSectorB = FALSE;
					HideDlogItem(ADFS_NewDisk_POP_SECTOR_ORDER);
				}
			} else {
				i_newDiskRec.format.sectorOrder = ND_MenuToType_Sector(
					GetControlValue(controlRef));

				//	sets ADFS_NewDisk_POP_FORMAT
				if (
					i_newDiskRec.format.sectorOrder != FSType_PRO
					&& i_newDiskRec.format.diskFileType == ND_Format_DISK_COPY
				) {
					if (i_origDiskFileType == ND_Format_DISK_COPY) {
						i_newDiskRec.format.diskFileType = ND_Format_RAW;
					} else {
						i_newDiskRec.format.diskFileType = i_origDiskFileType;
					}

					ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_FORMAT);
				}
			}
			break;
		}

		case ADFS_NewDisk_POP_OS: {
			if (convertType == DGI_ValueConvert_TO) {
				SetControlValue(
					controlRef, 
					ND_TypeToMenu_OS((FSType)i_newDiskRec.osType));
				
				//	doesn't depend
			} else {
				/*
					sets	ADFS_NewDisk_POP_SECTOR_ORDER
					sets	ADFS_NewDisk_POP_FORMAT
					sets	bytesOnDiskL

					affects ADFS_NewDisk_CHECK_NIBBLIZED
					affects ADFS_NewDisk_POP_VOL_SIZE
					affects ADFS_NewDisk_SIZE_TE
					affects ADFS_NewDisk_SIZE_UNITS
				*/
				
				i_newDiskRec.osType				= ND_MenuToType_OS(GetControlValue(controlRef));
				i_newDiskRec.format.sectorOrder	= ND_OSToSector((FSType)i_newDiskRec.osType);
				
				if (i_newDiskRec.osType != FSType_PRO) {
					i_newDiskRec.bytesOnDiskL = Gen_kBytesPerDisk;
				}

				if (
					i_newDiskRec.osType != FSType_PRO
					&& i_newDiskRec.format.diskFileType == ND_Format_DISK_COPY
				) {
					if (i_origDiskFileType == ND_Format_DISK_COPY) {
						i_newDiskRec.format.diskFileType = ND_Format_RAW;
					} else {
						i_newDiskRec.format.diskFileType = i_origDiskFileType;
					}

					ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_FORMAT);
				}

				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_CHECK_NIBBLIZED);
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_SECTOR_ORDER);
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_VOL_SIZE);
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_SIZE_TE);
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_SIZE_UNITS);
			}
			break;
		}

		case ADFS_NewDisk_POP_VOL_SIZE: {
			if (convertType == DGI_ValueConvert_TO) {
				if (i_newDiskRec.customSizeB) {
					SetControlValue(controlRef, ND_Custom);
				} else {
					SetControlValue(controlRef, 
						ND_TypeToMenu_VolSize(i_newDiskRec.bytesOnDiskL));
				}
				
				/*
					depends ADFS_NewDisk_CHECK_NIBBLIZED
					depends ADFS_NewDisk_POP_OS

					visible if 
						not ADFS_NewDisk_CHECK_NIBBLIZED
						and ADFS_NewDisk_POP_OS == ProDOS
				*/
				
				if (
					!i_showSizePopB
					&& i_newDiskRec.osType == FSType_PRO
					&& !i_newDiskRec.format.nibblizedB
				) {
					i_showSizePopB = TRUE;
					ShowDlogItem(ADFS_NewDisk_POP_VOL_SIZE);
				} else if (
					i_showSizePopB
					&& (
						i_newDiskRec.osType != FSType_PRO
						|| i_newDiskRec.format.nibblizedB
					)
				) {
					i_showSizePopB = FALSE;
					HideDlogItem(ADFS_NewDisk_POP_VOL_SIZE);
				}
			} else {
				short		menuItem		= GetControlValue(controlRef);
				long		bytesOnDiskL	=  ND_MenuToType_VolSize(menuItem);
				
				/*					
					sets	bytesOnDiskL
					sets	customSize
					
					affects ADFS_NewDisk_CHECK_NIBBLIZED
					affects ADFS_NewDisk_POP_SECTOR_ORDER
					affects ADFS_NewDisk_SIZE_TE
					affects ADFS_NewDisk_SIZE_UNITS
				*/
				
				//	if "Custom..."
				i_newDiskRec.customSizeB = bytesOnDiskL == kVolSizeCustom;
				if (!i_newDiskRec.customSizeB) {
					i_newDiskRec.bytesOnDiskL		= bytesOnDiskL;
				}
				
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_CHECK_NIBBLIZED);				
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_SECTOR_ORDER);				
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_SIZE_UNITS);				
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_SIZE_TE);
			}
			break;
		}

		case ADFS_NewDisk_SIZE_TE: {
			char		buf[32];
			long		value = i_newDiskRec.bytesOnDiskL;
			
			if (convertType == DGI_ValueConvert_TO) {

				value /= 1024;

				if (i_newDiskRec.isMB) {
					value /= 1024;
				}
				
				sprintf(buf, "%lu", value);
				SetDlogItemText(dlgItem, buf);

				/*
					depends ADFS_NewDisk_CHECK_NIBBLIZED
					depends ADFS_NewDisk_POP_OS
					depends customSize

					visible if
						not ADFS_NewDisk_CHECK_NIBBLIZED
						and ADFS_NewDisk_POP_OS == ProDOS
						and customSize == TRUE
				*/
				if (
					!i_showSizeUnitsB
					&& !i_newDiskRec.format.nibblizedB
					&& i_newDiskRec.osType == FSType_PRO
					&& i_newDiskRec.customSizeB
				) {
					i_showSizeUnitsB = TRUE;
					ShowDialogItemGroup(
						ADFS_NewDisk_SIZE_TE_Gr_FIRST, 
						ADFS_NewDisk_SIZE_TE_Gr_LAST);
				} else if (
					i_showSizeUnitsB
					&& (
						i_newDiskRec.format.nibblizedB
						|| i_newDiskRec.osType != FSType_PRO
						|| !i_newDiskRec.customSizeB
					)
				) {
					i_showSizeUnitsB = FALSE;
					HideDialogItemGroup(
						ADFS_NewDisk_SIZE_TE_Gr_FIRST, 
						ADFS_NewDisk_SIZE_TE_Gr_LAST);
				}				
				
				if (i_showSizeUnitsB) {
					SelectDlogItem(dlgItem);
					InvalDialogItem(dlgItem);
				}
			} else {
				/*
					sets	bytesOnDiskL
					
					affects ADFS_NewDisk_CHECK_NIBBLIZED
					affects ADFS_NewDisk_POP_SECTOR_ORDER
				*/
				
				GetDlogItemText(dlgItem, buf);
				sscanf(buf, "%lu", &value);

				value *= 1024;

				if (i_newDiskRec.isMB) {
					value *= 1024;
				}

				i_newDiskRec.bytesOnDiskL = value;
				
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_CHECK_NIBBLIZED);
				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_POP_SECTOR_ORDER);
			}
			break;
		}

		case ADFS_NewDisk_SIZE_UNITS: {
			if (convertType == DGI_ValueConvert_TO) {
				SetControlValue(
					controlRef, 
					ND_TypeToMenu_Units(i_newDiskRec.isMB));
				
				//	depends set by (and same as) TE
			} else {
				Boolean		isMB = ND_MenuToType_Units(
					GetControlValue(controlRef));

				/*
					sets	ADFS_NewDisk_SIZE_UNITS
					affects ADFS_NewDisk_SIZE_TE
				*/
				if (i_newDiskRec.isMB != isMB) {
					i_newDiskRec.isMB = isMB;
				}

				ValueConvert(DGI_ValueConvert_TO, ADFS_NewDisk_SIZE_TE);
			}
			break;
		}
	}
}

